// Keep everything in anonymous function, called on window load.
if(window.addEventListener) {
window.addEventListener('load', function () {
  var canvas, context, canvaso, contexto;

  // Variables de herramientas.
  var tool;
  var undoHistory = new Array();//Matriz que guarda imagenes para el deshacer
  var redoHistory = new Array();//Matriz que guarda imagenes para el rehacer
  var tool_default = 'cubo_pintura';
  var imagenFondo = "data:image/png;base64,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";
  rellenoR = 244;
  rellenoG = 237;
  rellenoB = 200;
  var fin = 'round';
  var grid = 20;//Tamao de rejilla
  var colorFondo = '#f3f0e7';//color de fondo y del borrador
  var radioPuntos = 1.5;//radio de crculos en punteado
  var densidad = 5;//factor que marca la distancia entre puntos respecto al grosor del punto en punteado2 y pincelPluma
  var espaciado = densidad * 5;//separacin entre pinceladas en pincel compuesto
  var pincel = 10;//radio del area donde se dibujan los puntos en pincel aleatorio y del borrador
  var anchopluma = 2;//en pincel pluma el ancho en el que hay trazos
  var grosorpluma = 1;//Grosor de los trazos en pincel pluma
  var canvasWidth;
  var canvasHeight;

  function init () {
    // Localiza el elemento canvas.
    canvaso = document.getElementById('imageView');
    if (!canvaso) {
      alert('Error: No puedo encontrar el elemento canvas!');
      return;
    }
   altoNuevo = canvaso.height;
   anchoNuevo = canvaso.width;
    if (!canvaso.getContext) {
      alert('Error: no canvas.getContext!');
      return;
    }

    // Toma el 2D canvas context.
    contexto = canvaso.getContext('2d');
    if (!contexto) {
      alert('Error: Necesitas un navegador ms moderno para poder ver esta aplicacin!');
      return;
    }
	
    // Crea contexto para canvas que est encima de todo
	
    canvasTop = document.getElementById('imageTop');	
    contextTop = canvasTop.getContext('2d');
    if (!contextTop) {
      alert('Error: failed to getContext!');
      return;
    }	
	
	canvasWidth = canvaso.width;
	canvasHeight = canvaso.height;
	
	//Ponemos fondo color
	//fondoColor();
	

    // Aade un canvas temporal.
    var container = canvaso.parentNode;
    canvas = document.createElement('canvas');
    if (!canvas) {
    alert('Error: No puedo crar un nuevo lienzo!');
    return;
    }
    canvas.id     = 'imageTemp';
    canvas.width  = canvaso.width;
    canvas.height = canvasHeight - 22;
	contTemp = document.getElementById('contenedor1');
    contTemp.appendChild(canvas);
    context = canvas.getContext('2d');



    // Activa la herramienta por defecto.
    if (tools[tool_default]) {
      tool = new tools[tool_default]();
    }
	//Pone por defecto el cursor cubo
	$('canvas').css("cursor","url(imagenes/cursores/cursor_cubo.png),auto");
	

    // Coloca mousedown, mousemove and mouseup event listeners.
    canvas.addEventListener('mousedown', ev_canvas, false);
    canvas.addEventListener('mousemove', ev_canvas, false);
    canvas.addEventListener('mouseup',  ev_canvas, false);
	canvas.addEventListener('dblclick',  ev_canvas, false);
	canvas.addEventListener('mouseout',  ev_canvas, false);
	canvaso.addEventListener('mouseup',  ev_canvas, false);
	
	//Eventos cuando se presiona una tecla
	window.onkeydown = function(evt) {
	evt = evt || window.event;
	if (evt.keyCode === 17) {
		keyControl = true;
		}
	}
	window.onkeyup = function(evt) {
	evt = evt || window.event;
		if (evt.keyCode === 17) {
			keyControl = false;
		}else if(evt.keyCode === 90 &&(keyControl)){//Deshace al presionar Ctrl + z
		undoDraw();
		}
	}
	//Evita seleccion azul
	document.onselectstart = function() {
	return false;
	}
	
	//si existe almacenada imagen de anterior trabajo en navegador la carga

	if (localStorage.dibujo_contraste_cromatico){
		if (confirm('Quieres recuperar el ltimo trabajo que se hizo con esta aplicacin?')) {
			localStorageImage = new Image();
			localStorageImage.addEventListener("load", function(event){
				localStorageImage.removeEventListener(event.type, arguments.callee, false);
				contexto.drawImage(localStorageImage,0,0);
				filtroByN(0);
			}, false);
			localStorageImage.src = localStorage.dibujo_contraste_cromatico;	
		}else{ponerFondo(imagenFondo);return}
	}else{ponerFondo(imagenFondo);return}

}
	//Carga imagen para patrn y lo crea.
	$('.textura').click(function (){
		var img = new Image();
			var direccion = $(this).attr('id');
			var textFondo = $(this).css('backgroundImage');
			img.src =  eval(direccion);
			img.onload = function(){
			var ptrn = context.createPattern(img,'repeat');
			relleno = ptrn;
		}
		$('#paleta3').css('backgroundImage', textFondo);
	})
	
	
	//Abre ventana para escribir texto en canvas
	$('#texto').click( function() {
		document.getElementById('text-controls').style.display = 'block';
		var textE1 = document.getElementById('text');
		//textE1.focus();
	})
	//Coloca texto al aceptar
	$('#botonTexto').click(function (){
		saveActions();
		contexto.save();
		contexto.fillStyle = colorFondo;
		contexto.fillRect(0,523, canvaso.width, 22);//rellena con color de fondo Si fuera una imagen habra que recargarla
		contexto.restore();
		titulo = document.getElementById('text').value;
		contexto.font = "12pt Verdana";
		contexto.fillStyle = "black";
		contexto.fillText(titulo, 20, 540);
		saveLocalStorage();
		document.getElementById('text-controls').style.display = 'none';
	})
	//Cierra ventana de colocar texto
	$('#botonTexCan').click(function (){document.getElementById('text-controls').style.display = 'none'});
	
	// Controlador de evento general. Esta funcin determina la
	// posicion relativa del ratn respecto al elemento canvas.Da las coordenadas del raton dentro del canvas
	function ev_canvas (ev) {
		if (ev.layerX || ev.layerX == 0) { // Firefox
		  ev._x = Math.round(ev.layerX / canvasScale);//multiplicamos por canvasScale para conpensar zoom
		  ev._y = Math.round(ev.layerY / canvasScale);
		} else if (ev.offsetX || ev.offsetX == 0) { // Opera
		  ev._x = Math.round(ev.offsetX / canvasScale);
		  ev._y = Math.round(ev.offsetY / canvasScale);
		}

		// Llama al manejador del evento de la herramienta
		var func = tool[ev.type];
		if (func) {
		  func(ev);
		}
	}

	// Esta funcin dibuja #imageTemp canvas encima de #imageView, despus
	// #imageTemp se limpia. Esto se realiza cada vez que el usuario
	// completa unaa operacin de dibujo.
	function img_update () {
  		saveActions();//Guarda copia de la imagen
		contexto.drawImage(canvas, 0, 0);//copia imagen temporal en definitiva
		context.clearRect(0, 0, canvas.width, canvas.height);//Limpia la imagen temporal
		redoHistory = [];
		saveLocalStorage();

	}
	//Function guarda imagen en array para poder deshacer
	function saveActions() {
		var imgData = document.getElementById("imageView").toDataURL("image/png");
		undoHistory.push(imgData);//guarda imagen en matriz deshacer
	}
	//Guarda imagen en el cache del navegador
	function saveLocalStorage(){
		var imgData = document.getElementById("imageView").toDataURL("image/png");
		localStorage.dibujo_contraste_cromatico = imgData;//guarda imagen en el cache del navegador
	}
	
	//Funcin pone fondo de color
	function fondoColor(){
	contexto.save();
	contexto.fillStyle = colorFondo;
	contexto.fillRect(0,0, canvaso.width, canvaso.height);
	contexto.restore();
	}
	
	//Funcion deshacer
	function undoDraw(){
		if(undoHistory.length > 0){
		var imgData = document.getElementById("imageView").toDataURL("image/png");
		redoHistory.push(imgData);//Guarda la imagen actual en array rehacer
        var undoImg = new Image();
			undoImg.onload = function(){//funcion para que le de tiempo a cargar la imagen
			contexto.drawImage(undoImg, 0,0);//copia imagen rescatada en canvas
			var imgData = document.getElementById("imageView").toDataURL("image/png");
			localStorage.dibujo_bodegon_punto = imgData;//guarda imagen en el cache del navegador
			};
			undoImg.src = undoHistory.pop();//saca ultima imagen guardada en el array y la guarda en undoImg
        } else if(undoHistory.length == 0){
		return
		}		
	}
	
	function redoDraw(){
		if(redoHistory.length > 0){
		saveActions();
        var redoImg = new Image();
			redoImg.onload = function(){//funcion para que le de tiempo a cargar la imagen
			contexto.drawImage(redoImg, 0,0);//copia imagen rescatada en canvas
			var imgData = document.getElementById("imageView").toDataURL("image/png");
			localStorage.dibujo_bodegon_punto = imgData;//guarda imagen en el cache del navegador
			};
			redoImg.src = redoHistory.pop();//saca ultima imagen guardada en el array y la guarda en redoImg
		}
	}
	
	
	//Pone una imagen como fondo
	function ponerFondo (imagenUrl){
		var img = new Image();
		img.src = imagenUrl;
		img.onload = function(){
		contexto.drawImage(img,0,0,750,523,0,0,750,523);
		}
	}

	//Control del boton limpiar
    $('#reset').click(function() {
        if (confirm('Se reiniciar el ejercicio y se perder la imagen actual. Ests seguro?')) {
		//canvaso.width = canvaso.width;
		undoHistory.length = 0;//vacia array con estados anteriores
		redoHistory.length = 0;
		contexto.save();
		contexto.fillStyle = colorFondo;
		contexto.fillRect(0,0, canvaso.width, canvaso.height);//rellena con color de fondo Si fuera una imagen habra que recargarla
		contexto.restore();
		ponerFondo(imagenFondo);
        }
    })

	//controlamos el boton guardar imagen
	$('#guardar').click (function() {
	var imgData = document.getElementById("imageView").toDataURL("image/png");
			var ventana = window.open('','imagenGenerada', "toolbar=no,location=no,status=no,menubar=no,resizable=yes,scrollbars=1");
			ventana.document.write('<html><head><title>Imagen Generada</title><meta http-equiv="content-type" content="text/html; charset=iso-8859-1"><link rel="stylesheet" type="text/css" href="css/dibujo_00.css" /> ');
			ventana.document.write('</head>');
			ventana.document.write('<body style="background-color:#F4EDC8">');
			ventana.document.write('<div  style="position:absolute; top:40px; left:50%; margin-left: -375px;  width:750px; height:545px; box-shadow: -2px 2px 10px 1px #999;"><div style="position:relative"></div><div><img src="');
			ventana.document.write(imgData);
			ventana.document.write('"width="750" height="545">');
			ventana.document.write('</div><br><div style="border-width:1;border-color:#901420;border-style:dashed;border-radius:15px; margin-top: 50px;margin-left: 150px; margin-right: 150px; margin-bottom: 20px; padding-left:20px; padding-right:20px; font-family:Verdana;font-size:16px;background-color:#f3f0e7;text-align:center">');
			ventana.document.write('<br><br>Para guardar la imagen haz clic sobre ella con el botn derecho. En el men que aparece selecciona: "Guardar imagen como..."');
			ventana.document.write('<br><br><button onclick="window.close();" class="botonVentana">Cerrar Ventana</button><br><br></div><br><br></div></body>');
			ventana.document.close();
	//window.open(document.getElementById("imageView").toDataURL("image/png"),'Canvas Export','height=450,width=625');
	})

	// tools Contiene la implementacion de cada herramienta
	var tools = {};
	//Controles para saber que botn se ha seleccionado
	$('#cubo_pintura').click (function (){
	tool = new tools['cubo_pintura']();
	$('canvas').css("cursor","url(imagenes/cursores/cursor_cubo.png),auto");
	})
	$('#colorPicker').click(function (){
	tool = new tools['cuentagotas']();
	$('canvas').css("cursor","url(imagenes/cursores/cursor_cuenta.png),auto");	
	})
	$('#deshacer').click (function() {
	undoDraw();
	})
	$('#rehacer').click (function() {
	redoDraw();
	})
	
    // Coloca una imagen encima
	function ponerImg(urlImg) {
	var img = new Image();
	img.src = urlImg;
		img.onload = function(){
		context = canvas.getContext('2d');
		context.drawImage(img,0,0,canvasWidth,canvasHeight);
		img_update();
		}
	}

    // Cuentagotas
	tools.cuentagotas = function () {
    var tool = this;
		this.mousedown = function (ev) {
		  pixelColor = contexto.getImageData( ev._x,ev._y,1,1);
		  if(pixelColor.data[0] + pixelColor.data[1] + pixelColor.data[2] == 0){
			  rellenoR = 1;
			  rellenoG = 1;
			  rellenoB = 1;		  
			}else{
			  rellenoR = pixelColor.data[0];
			  rellenoG = pixelColor.data[1];
			  rellenoB = pixelColor.data[2];
			}  
			  alfaColor();
			  $('#paleta').css('backgroundColor', relleno);
		};
	};


		//Cubo Pintura
		tools.cubo_pintura = function () {
		var tool = this;
		this.started = false;

			this.mousedown = function (ev) {
			curColor = { r:rellenoR, g:rellenoG, b:rellenoB};//toma color seleccionado en rgb
			saveActions();
			colorLayerData = contexto.getImageData(0, 0, canvasWidth, canvasHeight);//copia canvas a colorLayerData
			paintAt(ev._x, ev._y);
			saveLocalStorage();
			};
			floodFill = function (startX, startY, startR, startG, startB) {


					var newPos,
						x,
						y,
						pixelPos,
						reachLeft,
						reachRight,
						drawingBoundLeft = 0,
						drawingBoundTop = 0,
						drawingBoundRight = canvasWidth,
						drawingBoundBottom = canvasHeight-22;
						pixelStack = [[startX, startY]];

					while (pixelStack.length) {

						newPos = pixelStack.pop();
						x = newPos[0];
						y = newPos[1];

						// Get current pixel position
						pixelPos = (y * canvasWidth + x) * 4;

						// Go up as long as the color matches and are inside the canvas
						while (y >= drawingBoundTop && matchStartColor(pixelPos, startR, startG, startB)) {
							y -= 1;
							pixelPos -= canvasWidth * 4;
						}

						pixelPos += canvasWidth * 4;
						y += 1;
						reachLeft = false;
						reachRight = false;

						// Go down as long as the color matches and in inside the canvas
						while (y <= drawingBoundBottom && matchStartColor(pixelPos, startR, startG, startB)) {
							y += 1;

							colorPixel(pixelPos, curColor.r, curColor.g, curColor.b);

							if (x > drawingBoundLeft) {
								if (matchStartColor(pixelPos - 4, startR, startG, startB)) {
									if (!reachLeft) {
										// Add pixel to stack
										pixelStack.push([x - 1, y]);
										reachLeft = true;
									}
								} else if (reachLeft) {
									reachLeft = false;
								}
							}

							if (x < drawingBoundRight) {
								if (matchStartColor(pixelPos + 4, startR, startG, startB)) {
									if (!reachRight) {
										// Add pixel to stack
										pixelStack.push([x + 1, y]);
										reachRight = true;
									}
								} else if (reachRight) {
									reachRight = false;
								}
							}

							pixelPos += canvasWidth * 4;
						}
					}

				},


			clearCanvas = function () {
			contexto.clearRect(0, 0,canvasWidth, canvasHeight);
			}

			matchStartColor = function (pixelPos, startR, startG, startB) {

				r = colorLayerData.data[pixelPos];
				g = colorLayerData.data[pixelPos + 1];
				b = colorLayerData.data[pixelPos + 2];

				//Si el color que clickeamos es el negro
				if(r+g+b < 3){
				return false;
				}
				
				// If the current pixel matches the clicked color
				if (r === startR && g === startG && b === startB) {
					return true;
				}

				// If current pixel matches the new color
				if (r === curColor.r && g === curColor.g && b === curColor.b) {
					return false;
				}
				// If current pixel has different colour
				return false;
			}

			colorPixel = function (pixelPos, r, g, b, a) {

				colorLayerData.data[pixelPos] = r;
				colorLayerData.data[pixelPos + 1] = g;
				colorLayerData.data[pixelPos + 2] = b;
				colorLayerData.data[pixelPos + 3] = a !== undefined ? a : 255;
			};
			paintAt = function (startX, startY) {

				var pixelPos = (startY * canvasWidth + startX) * 4,
					r = colorLayerData.data[pixelPos],
					g = colorLayerData.data[pixelPos + 1],
					b = colorLayerData.data[pixelPos + 2],
					a = colorLayerData.data[pixelPos + 3];

				if (r === curColor.r && g === curColor.g && b === curColor.b) {
					// Return because trying to fill with the same color
					return;
				};

				floodFill(startX, startY, r, g, b);
				contexto.putImageData(colorLayerData, 0, 0);
			}
		}; //fin cubo pintura
	
		//Aumenta de tamao el escritorio
		document.getElementById('zoom_in').onclick = function() {
			if (canvasScale > 2) {
				return;	
			}
			canvasScale = canvasScale * scaleFactor;
			altoNuevo = Math.round(altoNuevo* scaleFactor);
			anchoNuevo = Math.round( anchoNuevo* scaleFactor);
			$('#imageView').css('height',altoNuevo);
			$('#imageView').css('width',anchoNuevo);
			$('.lienzo').css('height',altoNuevo);
			$('.lienzo').css('width',anchoNuevo);
			$('#imageTemp').css('height',altoNuevo-(22*canvasScale));
			$('#imageTemp').css('width',anchoNuevo);
			$('#imageTop').css('height',altoNuevo);
			$('#imageTop').css('width',anchoNuevo);
			objeto = document.getElementById('herram');
			objeto.style.height = altoNuevo +'px';
		}
	
		//Disminuye de tamao el escritorio
		document.getElementById('zoom_out').onclick = function() {
			if (canvasScale < 0.9) {
				return;	
			}
			canvasScale = canvasScale / scaleFactor;
			altoNuevo = Math.round(altoNuevo * (1 / scaleFactor));
			anchoNuevo = Math.round(anchoNuevo * (1 / scaleFactor));
			$('#imageView').css('height',altoNuevo);
			$('#imageView').css('width',anchoNuevo);
			$('.lienzo').css('height',altoNuevo);
			$('.lienzo').css('width',anchoNuevo);
			$('#imageTemp').css('height',altoNuevo-(22 * canvasScale));
			$('#imageTemp').css('width',anchoNuevo);
			$('#imageTop').css('height',altoNuevo);
			$('#imageTop').css('width',anchoNuevo);
			objeto = document.getElementById('herram');
			objeto.style.height = altoNuevo +'px';
		}
	
		//Transforma imagen importada a blanco y negro
		function filtroByN(flag){
		canvasData = contexto.getImageData(0, 0, canvasWidth, canvasHeight);
		pixel = canvasData.data;
			for (var x = 0; x < canvasWidth; x++) {
				for (var y = 0; y < canvasHeight; y++) {

					var idx = (x + y * canvasWidth) * 4;

					var r = pixel[idx + 0];
					var g = pixel[idx + 1];
					var b = pixel[idx + 2];
					var a = pixel[idx + 3];
					//var v = 0.2126*r + 0.7152*g + 0.0722*b;//filtro a grises
					
					
					var v = (r+g+b)/3;
					if (v <= 80){
					pixel[idx + 0] = 0;
					pixel[idx + 1] = 0;
					pixel[idx + 2] = 0;
					}else{
					pixel[idx + 0] = 255;
					pixel[idx + 1] = 255;
					pixel[idx + 2] = 255;
					pixel[idx + 3] = 0;
					}
				}
			}
			contextTop.putImageData(canvasData, 0, 0);
			imagenFondo = canvasTop.toDataURL("image/png");
			if(flag == 1){
			fondoColor();
			ponerFondo(imagenFondo);
			img_update ();
			}else{
			return;
			}
		}
	
		//Importa imagen desde archivo y la coloca como fondo del canvas fabric
		formEl = document.forms[0],
		fileEl = formEl.elements[0],

		formEl.onsubmit = fileEl.onchange = uploadHandler;
//
		function uploadHandler() {
		uploadFile(fileEl.files && fileEl.files[0]);
		return false;
		}		
		$('#subirArc').click(function(){fileEl.click()});
		
		//Sube archivo de imagen desde el ordenador
		function uploadFile(file) {
			if (file && file.type && file.type.match(/^image\/.*/)) {
				var fileReader = new FileReader();
				fileReader.onload = function(e) {
					var dataURL = e.target.result;
					var newImg = new Image();
					newImg.onload = ajustar;
					newImg.src = dataURL;
					contexto.clearRect(0, 0, canvas.width, canvas.height);	
					function ajustar(){
						
						alto = newImg.height;
						ancho = newImg.width;
						altoCanvas =canvaso.height-22;
						if(ancho/alto >= canvaso.width/altoCanvas){
							yCoor = Math.floor((altoCanvas-canvaso.width*alto/ancho)/2);
							altoPNG = Math.floor(canvaso.width*alto/ancho);
							contexto.drawImage(newImg,0,yCoor,canvaso.width,altoPNG);
							filtroByN(1);
							return
						}else{
							xCoor = Math.floor((canvaso.width-altoCanvas*ancho/alto)/2);
							anchoPNG = Math.floor(altoCanvas*ancho/alto);
							contexto.drawImage(newImg,xCoor,0,anchoPNG,altoCanvas);
							filtroByN(1);
						}
					}
				}
			fileReader.readAsDataURL(file);
			}return
		}

	init();

}, false); }
